<?php

namespace App\Http\Controllers;

use App\Models\PresensiMahasiswa;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\ValidationException;

class PresensiMahasiswaController extends Controller
{
    protected $presensiModel;

    public function __construct(PresensiMahasiswa $presensi)
    {
        $this->presensiModel = $presensi;
        $this->middleware('auth', ['only' => ['store', 'update', 'destroy', 'index', 'show']]);
    }

    // Metode untuk mendapatkan semua data presensi mahasiswa
    public function index()
    {
        // Ambil semua presensi dengan relasi yang diperlukan
        $presensi = $this->presensiModel->with(['jurusan', 'tahunAjar', 'semester', 'kelas', 'mahasiswa'])->get();
    
        // Kembalikan data presensi dengan relasi yang telah dimuat
        return response()->json([
            'message' => 'Data presensi ditemukan.',
            'data' => $presensi
        ]);
    }

    public function show($id)
    {
        // Temukan presensi berdasarkan ID dan sertakan relasi yang diperlukan
        $presensi = $this->presensiModel->with(['jurusan', 'tahunAjar', 'semester', 'kelas', 'mahasiswa'])->find($id);
        
        // Cek jika presensi tidak ditemukan
        if (!$presensi) {
            return response()->json(['message' => 'Record tidak ditemukan.'], 404);
        }
    
        // Kembalikan data presensi dengan relasi yang telah dimuat
        return response()->json([
            'message' => 'Record ditemukan.',
            'data' => $presensi
        ]);
    }

    // Metode untuk menambahkan atau memperbarui presensi mahasiswa
    // Metode untuk menambahkan atau memperbarui presensi mahasiswa
    public function store(Request $request)
    {
        // Validasi request
        $validator = Validator::make($request->all(), [
            'kd_jurusan' => 'required',
            'kd_tahun_ajar' => 'required',
            'kd_semester' => 'required',
            'kd_kelas' => 'required',
            'kd_matakuliah' => 'required',
            'tanggal' => 'required',
            'pertemuan' => 'required',
            'nim' => 'required',
            'kd_absen' => 'required',
        ]);
    
        // Jika validasi gagal
        if ($validator->fails()) {
            throw new ValidationException($validator);
        }
    
        // Mencari data presensi yang sudah ada
        $existingPresensi = $this->presensiModel->where([
            'kd_jurusan' => $request->kd_jurusan,
            'kd_tahun_ajar' => $request->kd_tahun_ajar,
            'kd_semester' => $request->kd_semester,
            'kd_kelas' => $request->kd_kelas,
            'kd_matakuliah' => $request->kd_matakuliah,
            'tanggal' => $request->tanggal,
            'pertemuan' => $request->pertemuan,
            'nim' => $request->nim,
        ])->first();
    
        if ($existingPresensi) {
            // Jika sudah ada, lakukan update
            $existingPresensi->kd_absen = $request->kd_absen;
            $existingPresensi->updated_at = now(); // Mengupdate waktu saat ini
            $existingPresensi->save();
    
            return response()->json($existingPresensi, 200); // Mengembalikan data yang diupdate
        } else {
            // Jika tidak ada, tambahkan presensi baru
            $presensi = $this->presensiModel->addPresensiMahasiswa($request->all());
            return response()->json($presensi, 201); // Mengembalikan data yang ditambahkan
        }
    }



    // Metode untuk memperbarui data presensi mahasiswa
    public function update(Request $request, $id)
    {
        // Validasi request
        $validator = Validator::make($request->all(), [
            'kd_jurusan' => 'required',
            'kd_tahun_ajar' => 'required',
            'kd_semester' => 'required',
            'kd_kelas' => 'required',
            'kd_matakuliah' => 'required',
            'tanggal' => 'required',
            'pertemuan' => 'required',
            'nim' => 'required',
            'kd_absen' => 'required',
        ]);

        // Jika validasi gagal
        if ($validator->fails()) {
            throw new ValidationException($validator);
        }

        // Memperbarui presensi
        try {
            $presensi = $this->presensiModel->updatePresensiMahasiswa($id, $request->all());
            return response()->json($presensi);
        } catch (\Exception $e) {
            return response()->json(['message' => $e->getMessage()], 404);
        }
    }

    // Metode untuk menghapus data presensi mahasiswa
    public function destroy($nim, $kd_matakuliah, $kd_jurusan, $kd_tahun_ajar, $kd_semester, $kd_kelas, $pertemuan)
    {
        try {
            $this->presensiModel->destroyPresensiMahasiswa($nim, $kd_matakuliah, $kd_jurusan, $kd_tahun_ajar, $kd_semester, $kd_kelas, $pertemuan);
            return response()->json(['message' => 'Record berhasil dihapus.'], 200);
        } catch (\Exception $e) {
            return response()->json(['message' => $e->getMessage()], 404);
        }
    }
}
